source("BetaBinomial.R")

.double.xmax <- .Machine$double.xmax



MaxEntMultinomial3.fast <- function(
    n.ij,
    n.ik,
    n.jk,
    sample=NULL,
    start=NULL,
    warn=TRUE,
    eps=1e-3,
    maxiter=200,
    trace=FALSE
    ){
    if(!is.loaded("tripropscal")) dyn.load(paste("tripropscal", .Platform$dynlib.ext, sep=""))
    # Check arguments
    n.ij <- as.matrix(n.ij)
    n.ik <- as.matrix(n.ik)
    n.jk <- as.matrix(n.jk)

    mycall <- match.call()
    if(!missing(sample)){
        sample.arg <- deparse(mycall$sample)
        sample <- match(sample.arg,c("n.ij","n.ik","n.jk"))
        if(!is.finite(sample)){
            sample <- as.numeric(sample)
            if(!(sample %in% 1:3))
                stop("Argument 'sample' should be either 'n.ij','n.ik','n.jk',1,2,or,3")
        }
    }
        
    if(nrow(n.ij) != nrow(n.ik)
            ||
            ncol(n.ij) != nrow(n.jk)
            ||
            ncol(n.ik) != ncol(n.jk)
    ) stop("Dimensions do not match")
    # Setup the data
    
    I <- nrow(n.ij)
    J <- ncol(n.ij)
    K <- ncol(n.jk)
    IJK <- I*J*K
    
    p.ij0 <- n.ij/sum(n.ij)
    p.i0k <- n.ik/sum(n.ik)
    p.0jk <- n.jk/sum(n.jk)
    
    ijk <- as.matrix(expand.grid(1:I,1:J,1:K))
    ij <-  ijk[,c(1,2)]
    ik <-  ijk[,c(1,3)]  
    jk <-  ijk[,c(2,3)]  
    k <- ijk[,3]
    
    # Initialize the algorithm
    if(length(start)){
      if(!all(dim(start)==c(I,J,K))) stop("Incorrect starting array")
      if(!inherits(start,"MultinomialDist") || !length(attr(start,"params"))) {
        A.ij <- matrix(1,I,J)
        B.ik <- matrix(1,I,K)
        C.jk <- matrix(1,J,K)
        wrong.parms <- TRUE && warn
      }
      else {
        A.ij <- exp(attr(start,"params")$alpha)
        B.ik <- exp(attr(start,"params")$beta)
        C.jk <- exp(attr(start,"params")$gamma)
        wrong.parms <- attr(start,"wrong.parms")
      }
      if(trace) cat("\nUsing starting values")
      pi.ijk <- start
    }
    else {
      pi.ijk <- start <- array(1,dim=c(I,J,K))
      
      A.ij <- matrix(1,I,J)
      B.ik <- matrix(1,I,K)
      C.jk <- matrix(1,J,K)
      wrong.parms <- FALSE
    }        
    if(wrong.parms) warning("Resulting structural parameters may be incorrect")

    converged <- FALSE
    res <- .C("tripropscal",
      as.integer(I),
      as.integer(J),
      as.integer(K),
      as.double(p.ij0),
      as.double(p.i0k),
      as.double(p.0jk),
      pi.ijk = as.double(pi.ijk),
      A.ij = as.double(A.ij),
      B.ik = as.double(B.ik),
      C.jk = as.double(C.jk),
      as.double(eps),
      as.integer(maxiter),
      converged = as.integer(converged),
      iter = as.integer(1),
      as.integer(trace)
      )    

    pi.ijk[] <- res$pi.ijk
    A.ij[] <- res$A.ij
    B.ik[] <- res$B.ik
    C.jk[] <- res$C.jk
    converged <- res$converged
    iter <- res$iter

    Dimnames <- list()
    if(length(dimnames(n.ij)[1])) Dimnames[1] <- dimnames(n.ij)[1]
    if(length(dimnames(n.ij)[2])) Dimnames[2] <- dimnames(n.ij)[2]
    
    if(!length(Dimnames[1]) && length(dimnames(n.ik)[1])) Dimnames[1] <- dimnames(n.ik)[1]
    if(length(dimnames(n.ik)[2])) Dimnames[3] <- dimnames(n.ik)[2]
    
    if(!length(Dimnames[2]) && length(dimnames(n.jk)[1])) Dimnames[2] <- dimnames(n.ik)[1]
    if(!length(Dimnames[3]) && length(dimnames(n.jk)[2])) Dimnames[3] <- dimnames(n.ik)[2]

    if(length(Dimnames))
      dimnames(pi.ijk) <- Dimnames
        
    if(length(sample)==0){
        n <- mean(sapply(list(n.ij,n.jk,n.jk),sum))
        m <- NULL
        }
    else {
        n <- mean(sapply(list(n.ij,n.jk,n.jk)[-sample],sum))
        m <- mean(sapply(list(n.ij,n.jk,n.jk)[sample],sum))
        }
    return(structure(pi.ijk,
                    n=n,
                    m=m,
                    params=list(
                        alpha=log(A.ij),
                        beta=log(B.ik),
                        gamma=log(C.jk)
                        ),
                    iterations=iter,
                    converged=converged,
                    wrong.parms=wrong.parms,
                    class=c("MaxEntMultinomialDist","MultinomialDist")
                    ))
}


MaxEntMultinomial3.slow <- function(
    n.ij,
    n.ik,
    n.jk,
    sample=NULL,
    start=NULL,
    warn=TRUE,
    eps=1e-3,
    maxiter=200,
    trace=TRUE
    ){
    # Check arguments
    n.ij <- as.matrix(n.ij)
    n.ik <- as.matrix(n.ik)
    n.jk <- as.matrix(n.jk)

    mycall <- match.call()
    if(!missing(sample)){
        sample.arg <- deparse(mycall$sample)
        sample <- match(sample.arg,c("n.ij","n.ik","n.jk"))
        if(!is.finite(sample)){
            sample <- as.numeric(sample)
            if(!(sample %in% 1:3))
                stop("Argument 'sample' should be either 'n.ij','n.ik','n.jk',1,2,or,3")
        }
    }
        
    if(nrow(n.ij) != nrow(n.ik)
            ||
            ncol(n.ij) != nrow(n.jk)
            ||
            ncol(n.ik) != ncol(n.jk)
    ) stop("Dimensions do not match")
    # Setup the data
    
    I <- nrow(n.ij)
    J <- ncol(n.ij)
    K <- ncol(n.jk)
    IJK <- I*J*K
    
    p.ij0 <- n.ij/sum(n.ij)
    p.i0k <- n.ik/sum(n.ik)
    p.0jk <- n.jk/sum(n.jk)
    
    ijk <- as.matrix(expand.grid(1:I,1:J,1:K))
    ij <-  ijk[,c(1,2)]
    ik <-  ijk[,c(1,3)]  
    jk <-  ijk[,c(2,3)]  
    k <- ijk[,3]
    
    # Initialize the algorithm
    if(length(start)){
      if(!all(dim(start)==c(I,J,K))) stop("Incorrect starting array")
      if(!inherits(start,"MultinomialDist") || !length(attr(start,"params"))) {
        A.ij <- matrix(1,I,J)
        B.ik <- matrix(1,I,K)
        C.jk <- matrix(1,J,K)
        wrong.parms <- TRUE && warn
      }
      else {
        A.ij <- exp(attr(start,"params")$alpha)
        B.ik <- exp(attr(start,"params")$beta)
        C.jk <- exp(attr(start,"params")$gamma)
        wrong.parms <- attr(start,"wrong.parms")
      }
      if(trace) cat("\nUsing starting values")
      pi.ijk <- start
    }
    else {
      pi.ijk <- start <- array(1,dim=c(I,J,K))
      
      A.ij <- matrix(1,I,J)
      B.ik <- matrix(1,I,K)
      C.jk <- matrix(1,J,K)
      wrong.parms <- FALSE
    }        
    if(wrong.parms) warning("Resulting structural parameters may be incorrect")
    
    iter <- 0
    converged <- FALSE
    criterion <- Inf
    
    # Start it
    for(iter in 1:maxiter) {
            last.pi.ijk <- pi.ijk
    # Rescale along i an j
            pi.ij0 <- apply(pi.ijk,c(1,2),sum)
            q.ij0 <- ifelse(pi.ij0==0,0,p.ij0/pi.ij0)
            A.ij <- q.ij0*A.ij
            pi.ijk <- sweep(pi.ijk,c(1,2),q.ij0,"*")
    # Rescale along i an k
            pi.i0k <- apply(pi.ijk,c(1,3),sum)
            q.i0k <- ifelse(pi.i0k==0,0,p.i0k/pi.i0k)
            B.ik <- q.i0k*B.ik
            pi.ijk <- sweep(pi.ijk,c(1,3),q.i0k,"*")
    # Rescale along j an k
            pi.0jk <- apply(pi.ijk,c(2,3),sum)
            q.0jk <- ifelse(pi.0jk==0,0,p.0jk/pi.0jk)
            C.jk <- q.0jk*C.jk
            pi.ijk <- sweep(pi.ijk,c(2,3),q.0jk,"*")
                                    
    # Check convergence
            criterion <- IJK*max(abs(pi.ijk - last.pi.ijk))
            if(trace) cat(paste("\nItereration ",iter,": ",
                    criterion,
                    sep=""))
            if(criterion < eps) {
                    converged <- TRUE
                    break
                    }
    }
    if(!(converged)) {
            warning("Iterative scaling did not converge")
            }
    else if(trace) cat("\nConverged\n")
        
    Dimnames <- list()
    if(length(dimnames(n.ij)[1])) Dimnames[1] <- dimnames(n.ij)[1]
    if(length(dimnames(n.ij)[2])) Dimnames[2] <- dimnames(n.ij)[2]
    
    if(!length(Dimnames[1]) && length(dimnames(n.ik)[1])) Dimnames[1] <- dimnames(n.ik)[1]
    if(length(dimnames(n.ik)[2])) Dimnames[3] <- dimnames(n.ik)[2]
    
    if(!length(Dimnames[2]) && length(dimnames(n.jk)[1])) Dimnames[2] <- dimnames(n.ik)[1]
    if(!length(Dimnames[3]) && length(dimnames(n.jk)[2])) Dimnames[3] <- dimnames(n.ik)[2]

    if(length(Dimnames))
      dimnames(pi.ijk) <- Dimnames
        
    if(length(sample)==0){
        n <- mean(sapply(list(n.ij,n.jk,n.jk),sum))
        m <- NULL
        }
    else {
        n <- mean(sapply(list(n.ij,n.jk,n.jk)[-sample],sum))
        m <- mean(sapply(list(n.ij,n.jk,n.jk)[sample],sum))
        }
    return(structure(pi.ijk,
                    n=n,
                    m=m,
                    params=list(
                        alpha=log(A.ij),
                        beta=log(B.ik),
                        gamma=log(C.jk)
                        ),
                    iterations=iter,
                    converged=converged,
                    wrong.parms=wrong.parms,
                    class=c("MaxEntMultinomialDist","MultinomialDist")
                    ))
}


MaxEntMultinomial3 <- MaxEntMultinomial3.fast

DirichletParms <-function(p,
        theta.0=NULL,
        p.eps=1e-7,
        method=c("remove","substitute"),
        lower=min(interval),
        upper=max(interval),
        interval=c(1,max(attr(p,"n"),2*R)),
        eps=1e-10,
        maxiter=1000
        ){
    method <- match.arg(method)
    
    if(method=="remove")
      pi.r <- p[p >= p.eps]
    if(method=="substitute"){
      pi.r <- p
      pi.r[p < p.eps] <- pi.r
      }
    pi.r <- pi.r/sum(pi.r)  
    R <- length(pi.r)
    
    entropy <- function(theta.0){
        theta.r <- pi.r*theta.0
        if(any(theta.r < 0)) return(.double.xmax)
        res <- (theta.r - 1)*digamma(theta.r) -lgamma(theta.r )
        -sum(res) - lgamma(theta.0) + (theta.0 - R)*digamma(theta.0)
        }
    grad.entropy <- function(theta.0){
        theta.r <- pi.r*theta.0
        res <- (theta.r-1)*pi.r*trigamma(theta.r)
        (theta.0-R)*trigamma(theta.0) - sum(res)
        }
    uninformative <- FALSE
    if(missing(theta.0) && FALSE) {
        solution <- try(uniroot(grad.entropy,lower=lower,upper=upper,tol=eps,maxiter=maxiter),silent=TRUE)
        if(class(solution)=="try-error"){
          err.msg <- paste("There seems to be a problem with your search limits or a numerical instability condition:",
          "\n\tgradient-of-entropy(",lower,") = ",grad.entropy(lower),
          "\n\tgradient-of-entropy(",upper,") = ",grad.entropy(upper),
          "\n\tIt may be a good idea to use a different search interval",
          " or another value for p.eps (current value: ",p.eps,").",
          "\n\tVery small values for pi.ijk cause huge values of the trigamma function",
          "\n\twhich is involved in finding entropy maximizing parameters.",
          "\n\n\tUsing 'PlotDirichletEntropy' or 'PlotDirichletGradEntropy' with",
          "\n\tdifferent settings of p.eps may be of help here.",
          sep="")
          stop(err.msg)
        }
        theta.0 <- solution$root
        uninformative <- TRUE
    }
    if(missing(theta.0) && TRUE) {
        solution <- optimize(entropy,lower=lower,upper=upper,tol=eps,maximum=TRUE)
        theta.0 <- solution$maximum
        uninformative <- TRUE
      }
    res <- p*theta.0
    if(method=="substitute")
      res[p < p.eps] <- p.eps*theta.0
    if(method=="remove")
      res[p < p.eps] <- 0
    return(structure(
            .Data = res,
            theta.0 = theta.0,
            entropy = entropy(theta.0),
            class=c(
                if(uninformative) "MaxEntDirichletParms"
                    else "InformativeDirichletParms",
                "DirichletParms"    
                )
        ))
}

DirichletToBetaItv <- function(theta.r,coverage=.95){
  alpha = (1-coverage)/2
  theta.0 <- sum(theta.r)
  upper <- qbeta(1-alpha,shape1=theta.r,shape2=theta.0-theta.r)
  lower <- qbeta(alpha,shape1=theta.r,shape2=theta.0-theta.r)
  dim(upper) <- dim(theta.r)
  dim(lower) <- dim(theta.r)
  dimnames(upper) <- dimnames(theta.r)
  dimnames(lower) <- dimnames(theta.r)
  return(structure(list(
        upper=upper,
        lower=lower
    ),
    coverage=coverage
    ))
}

DirichletToBetaBinomItv <- function(theta.r,n=1,coverage=.95,method=NULL) {
  if(missing(method)) {
    if(n/length(theta.r)>10) method <- 2
    else method <- 1
    }
  alpha = (1-coverage)/2
  theta.0 <- sum(theta.r)
  upper <- qbetabinom(1-alpha,size=n,phi=theta.r,psi=theta.0-theta.r,lower.tail=TRUE,method=method)
  lower <- qbetabinom(alpha,size=n,phi=theta.r,psi=theta.0-theta.r,lower.tail=TRUE,method=method)
  dim(upper) <- dim(theta.r)
  dim(lower) <- dim(theta.r)
  dimnames(upper) <- dimnames(theta.r)
  dimnames(lower) <- dimnames(theta.r)
  return(structure(list(
        upper=upper,
        lower=lower
    ),
    coverage=coverage
    ))
}

